package nemosofts.notes.app.adapters;

import android.annotation.SuppressLint;
import android.content.Context;
import android.content.res.ColorStateList;
import android.graphics.BitmapFactory;
import android.graphics.Color;
import android.net.Uri;
import android.os.Handler;
import android.os.Looper;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.TextView;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import androidx.nemosofts.material.ImageViewRound;
import androidx.nemosofts.theme.ThemeEngine;
import androidx.recyclerview.widget.RecyclerView;

import com.google.android.material.card.MaterialCardView;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Locale;
import java.util.Timer;
import java.util.TimerTask;

import nemosofts.notes.app.R;
import nemosofts.notes.app.item.Note;
import nemosofts.notes.app.listeners.NotesListener;

public class NoteAdapter extends RecyclerView.Adapter<NoteAdapter.NoteViewHolder> {

    private List<Note> notes;
    private final NotesListener notesListener;
    private Timer timer;
    private final List<Note> noteSource;
    private final boolean isDarkMode;
    private final int accentText;

    public NoteAdapter(Context context, List<Note> notes, NotesListener notesListener) {
        this.notes = notes;
        this.notesListener = notesListener;
        this.noteSource = notes;
        isDarkMode = new ThemeEngine(context).getIsDark();
        accentText = ContextCompat.getColor(context, R.color.colorWhite);
    }

    @NonNull
    @Override
    public NoteViewHolder onCreateViewHolder(@NonNull ViewGroup parent, int viewType) {
        View itemView = LayoutInflater.from(parent.getContext()).inflate(R.layout.row_note, parent, false);
        return new NoteViewHolder(itemView);
    }

    @Override
    public void onBindViewHolder(@NonNull NoteViewHolder holder, int position) {
        final Note note = notes.get(position);
        holder.layoutNote.setOnClickListener(view -> notesListener.onNoteClicked(notes.get(position), position));
        holder.itemNew.setVisibility(position == 0 ? View.VISIBLE : View.GONE);

        holder.textTitle.setText(note.getTitle());
        if (note.getSubtitle().trim().isEmpty()) {
            holder.textSubtitle.setVisibility(View.GONE);
        } else {
            holder.textSubtitle.setText(note.getSubtitle());
            holder.textSubtitle.setVisibility(View.VISIBLE);
        }
        holder.textDateTime.setText(note.getDateTime());
        holder.linkFlag.setVisibility(
                note.getWebLink() != null && !note.getWebLink().trim().isEmpty()
                        ? View.VISIBLE : View.GONE
        );

        int defaultColor = isDarkMode ? Color.parseColor("#ECECEC") : Color.parseColor("#121212");
        int cardColor = defaultColor;
        String noteColor = note.getColor();
        if (noteColor != null && !noteColor.trim().isEmpty()) {
            try {
                if (!"#333333".equals(noteColor)) {
                    cardColor = Color.parseColor(noteColor);
                }
            } catch (IllegalArgumentException ignored) {
                cardColor = defaultColor;
            }
        }

        holder.layoutNote.setCardBackgroundColor(cardColor);
        holder.colorStripe.setBackgroundColor(cardColor);

        boolean useDarkText = !isColorDark(cardColor);
        int primaryText = useDarkText ? Color.parseColor("#161616") : Color.parseColor("#F5F5F5");
        int secondaryText = useDarkText ? Color.parseColor("#6D6D6D") : Color.parseColor("#E9A0A0A0");

        holder.textTitle.setTextColor(primaryText);
        holder.textSubtitle.setTextColor(secondaryText);
        holder.textDateTime.setTextColor(primaryText);
        holder.textDateTime.setBackgroundTintList(ColorStateList.valueOf(adjustAlpha(cardColor, 0.2f)));
        if (note.isReminderEnabled() && note.getReminderTime() > 0) {
            String formattedTime = new SimpleDateFormat("dd MMM, HH:mm", Locale.getDefault())
                    .format(new Date(note.getReminderTime()));
            holder.reminderFlag.setText(formattedTime);
            holder.reminderFlag.setVisibility(View.VISIBLE);
            holder.reminderFlag.setTextColor(accentText);
        } else {
            holder.reminderFlag.setVisibility(View.GONE);
        }
        if (note.getWebLink() != null && !note.getWebLink().trim().isEmpty()) {
            holder.linkFlag.setVisibility(View.VISIBLE);
            holder.linkFlag.setTextColor(accentText);
        } else {
            holder.linkFlag.setVisibility(View.GONE);
        }
        if (note.getAudioPath() != null && !note.getAudioPath().trim().isEmpty()) {
            holder.audioFlag.setVisibility(View.VISIBLE);
            holder.audioFlag.setTextColor(accentText);
        } else {
            holder.audioFlag.setVisibility(View.GONE);
        }

        String imagePath = note.getImagePath();
        if (imagePath != null && !imagePath.trim().isEmpty()) {
            if (imagePath.startsWith("content://")) {
                holder.imageNote.setImageURI(Uri.parse(imagePath));
            } else {
                holder.imageNote.setImageBitmap(BitmapFactory.decodeFile(imagePath));
            }
            holder.imageNote.setVisibility(View.VISIBLE);
        } else {
            holder.imageNote.setVisibility(View.GONE);
        }
    }

    @Override
    public int getItemCount() {
        return notes.size();
    }

    @Override
    public int getItemViewType(int position) {
        return position;
    }

    static class NoteViewHolder extends RecyclerView.ViewHolder {

        TextView textTitle;
        TextView textSubtitle;
        TextView textDateTime;
        TextView itemNew;
        MaterialCardView layoutNote;
        ImageViewRound imageNote;
        View colorStripe;
        TextView linkFlag;
        TextView reminderFlag;
        TextView audioFlag;

        public NoteViewHolder(@NonNull View itemView) {
            super(itemView);

            itemNew = itemView.findViewById(R.id.item_new);
            textTitle = itemView.findViewById(R.id.item_textTitlem);
            textSubtitle = itemView.findViewById(R.id.item_textSubTitle);
            textDateTime = itemView.findViewById(R.id.item_textDateTime);
            layoutNote = itemView.findViewById(R.id.item_layoutNote);
            imageNote = itemView.findViewById(R.id.item_imageNote);
            colorStripe = itemView.findViewById(R.id.item_colorStripe);
            linkFlag = itemView.findViewById(R.id.item_linkFlag);
            reminderFlag = itemView.findViewById(R.id.item_reminderFlag);
            audioFlag = itemView.findViewById(R.id.item_audioFlag);
        }

    }

    public void searchNote(final String searchKeyword) {
        timer = new Timer();
        timer.schedule(new TimerTask() {
            @SuppressLint("NotifyDataSetChanged")
            @Override
            public void run() {
                if (searchKeyword.trim().isEmpty()) {
                    notes = noteSource;
                } else {
                    ArrayList<Note> temp = new ArrayList<>();
                    for (Note note : noteSource) {
                        if (note.getTitle().toLowerCase().contains(searchKeyword.toLowerCase())
                                || note.getSubtitle().toLowerCase().contains(searchKeyword.toLowerCase())
                                || note.getNoteText().toLowerCase().contains(searchKeyword.toLowerCase())) {
                            temp.add(note);
                        }
                    }
                    notes = temp;
                }

                new Handler(Looper.getMainLooper()).post(() -> notifyDataSetChanged());
            }
        }, 500);
    }

    public void cancelTimer() {
        if (timer != null) {
            timer.cancel();
        }
    }

    private boolean isColorDark(int color) {
        double darkness = 1 - (0.299 * Color.red(color) + 0.587 * Color.green(color) + 0.114 * Color.blue(color)) / 255;
        return darkness >= 0.5;
    }

    private int adjustAlpha(int color, float factor) {
        int alpha = Math.round(Color.alpha(color) * factor);
        return Color.argb(alpha, Color.red(color), Color.green(color), Color.blue(color));
    }
}
